# Step 5: Using State for Interactive Filtering

import streamlit as st
import pandas as pd
import numpy as np

st.subheader("State for Data Filtering")

# --- Initialize Filter State ---
if 'filter_value' not in st.session_state:
    st.session_state.filter_value = 0.5

# --- Sample Data (must be defined outside the conditional block) ---
@st.cache_data
def load_sample_data():
    return pd.DataFrame({
        'data_id': range(1, 11),
        'value': np.linspace(0, 1, 10),
        'category': np.random.choice(['A', 'B', 'C'], 10)
    })

sample_df = load_sample_data()

# --- Filter Widget ---
st.write("Filter data based on 'value':")
new_filter_value = st.slider(
    "Minimum Value Threshold", 
    0.0, 1.0, 
    value=st.session_state.filter_value,
    step=0.1,
    key='filter_slider' # Links widget value to st.session_state.filter_value
)

# --- Apply Filter and Display ---
filtered_df = sample_df[sample_df['value'] >= st.session_state.filter_value]

st.dataframe(filtered_df, use_container_width=True)

st.info(f"Displaying {len(filtered_df)} out of {len(sample_df)} rows (Filtered by: value >= {st.session_state.filter_value})")

# KEY CONCEPT: Even if other parts of the app cause a rerun, the slider's 
# position (st.session_state.filter_value) is preserved, ensuring a consistent 
# user experience.
